var _a, _b, _c, _d;
import { c as create, p as persist, a as createJSONStorage, b as chromeStorage, r as registerHandler, i as initBackgroundMessageListener } from "./storage-B2ZD4aVa.js";
create()(
  persist(
    (set, get) => ({
      connectedDapps: {},
      addConnection: (dapp) => {
        set((state2) => ({
          connectedDapps: {
            ...state2.connectedDapps,
            [dapp.origin]: dapp
          }
        }));
      },
      removeConnection: (origin) => {
        set((state2) => {
          const { [origin]: _, ...rest } = state2.connectedDapps;
          return { connectedDapps: rest };
        });
      },
      isConnected: (origin) => {
        return !!get().connectedDapps[origin];
      },
      getConnection: (origin) => {
        return get().connectedDapps[origin] || null;
      },
      hasPermission: (origin, permission) => {
        const dapp = get().connectedDapps[origin];
        return (dapp == null ? void 0 : dapp.permissions.includes(permission)) ?? false;
      },
      updatePermissions: (origin, permissions) => {
        set((state2) => {
          const dapp = state2.connectedDapps[origin];
          if (!dapp) return state2;
          return {
            connectedDapps: {
              ...state2.connectedDapps,
              [origin]: {
                ...dapp,
                permissions
              }
            }
          };
        });
      },
      getAllConnections: () => {
        return Object.values(get().connectedDapps);
      },
      clearAllConnections: () => {
        set({ connectedDapps: {} });
      }
    }),
    {
      name: "p01-connections",
      storage: createJSONStorage(() => chromeStorage)
    }
  )
);
async function getConnectedSites() {
  var _a2;
  try {
    const result = await chrome.storage.local.get("p01-connections");
    const data = result["p01-connections"];
    if (data) {
      const parsed = typeof data === "string" ? JSON.parse(data) : data;
      return ((_a2 = parsed.state) == null ? void 0 : _a2.connectedDapps) || {};
    }
  } catch (error) {
    console.error("Failed to get connected sites:", error);
  }
  return {};
}
async function addConnectedSite(dapp) {
  try {
    const sites = await getConnectedSites();
    sites[dapp.origin] = dapp;
    await chrome.storage.local.set({
      "p01-connections": JSON.stringify({
        state: { connectedDapps: sites },
        version: 0
      })
    });
  } catch (error) {
    console.error("Failed to add connected site:", error);
    throw error;
  }
}
async function removeConnectedSite(origin) {
  try {
    const sites = await getConnectedSites();
    delete sites[origin];
    await chrome.storage.local.set({
      "p01-connections": JSON.stringify({
        state: { connectedDapps: sites },
        version: 0
      })
    });
  } catch (error) {
    console.error("Failed to remove connected site:", error);
    throw error;
  }
}
async function isSiteConnected(origin) {
  const sites = await getConnectedSites();
  return !!sites[origin];
}
async function siteHasPermission(origin, permission) {
  const sites = await getConnectedSites();
  const site = sites[origin];
  return (site == null ? void 0 : site.permissions.includes(permission)) ?? false;
}
const state = {
  pendingApprovals: /* @__PURE__ */ new Map()
};
const SUBSCRIPTION_CHECK_ALARM = "p01-subscription-check";
const SUBSCRIPTION_CHECK_INTERVAL_MINUTES = 15;
let pendingPayments = [];
function isPaymentDue(sub) {
  return sub.isActive && sub.nextPayment <= Date.now();
}
function calculateNextPayment(sub) {
  return {
    amount: sub.amount,
    timestamp: sub.nextPayment
  };
}
async function initSubscriptionScheduler() {
  await chrome.alarms.create(SUBSCRIPTION_CHECK_ALARM, {
    periodInMinutes: SUBSCRIPTION_CHECK_INTERVAL_MINUTES,
    delayInMinutes: 1
  });
  console.log("[Stream Secure] Subscription scheduler initialized");
}
async function checkDueSubscriptions() {
  var _a2, _b2;
  console.log("[Stream Secure] Checking for due subscriptions...");
  try {
    const result = await chrome.storage.local.get("p01-subscriptions");
    const storedData = result["p01-subscriptions"] ? JSON.parse(result["p01-subscriptions"]) : null;
    if (!((_a2 = storedData == null ? void 0 : storedData.state) == null ? void 0 : _a2.subscriptions)) {
      console.log("[Stream Secure] No subscriptions found");
      return;
    }
    const subscriptions = storedData.state.subscriptions;
    const dueSubscriptions = subscriptions.filter(isPaymentDue);
    if (dueSubscriptions.length === 0) {
      console.log("[Stream Secure] No payments due");
      return;
    }
    console.log(`[Stream Secure] ${dueSubscriptions.length} payment(s) due`);
    const walletResult = await chrome.storage.local.get("p01-wallet");
    const walletState = walletResult["p01-wallet"] ? JSON.parse(walletResult["p01-wallet"]) : null;
    if (!((_b2 = walletState == null ? void 0 : walletState.state) == null ? void 0 : _b2.publicKey)) {
      console.log("[Stream Secure] Wallet not initialized, skipping payments");
      return;
    }
    for (const sub of dueSubscriptions) {
      await notifyPendingPayment(sub);
      if (!pendingPayments.find((p) => p.subscriptionId === sub.id)) {
        pendingPayments.push({
          subscriptionId: sub.id,
          scheduledTime: sub.nextPayment,
          attempts: 0
        });
      }
    }
    const isWalletUnlocked = await checkWalletUnlocked();
    if (isWalletUnlocked) {
      await processPendingPayments();
    }
  } catch (error) {
    console.error("[Stream Secure] Error checking subscriptions:", error);
  }
}
async function checkWalletUnlocked() {
  try {
    const result = await chrome.storage.session.get("p01-wallet-unlocked");
    return result["p01-wallet-unlocked"] === true;
  } catch {
    return false;
  }
}
async function processPendingPayments() {
  if (pendingPayments.length === 0) {
    return;
  }
  console.log(`[Stream Secure] Processing ${pendingPayments.length} pending payment(s)`);
  try {
    await chrome.runtime.sendMessage({
      type: "PROCESS_SUBSCRIPTION_PAYMENTS",
      subscriptionIds: pendingPayments.map((p) => p.subscriptionId)
    });
  } catch {
    console.log("[Stream Secure] Could not send message to popup");
  }
}
async function notifyPendingPayment(sub) {
  try {
    const permission = await chrome.permissions.contains({
      permissions: ["notifications"]
    });
    if (!permission) {
      console.log("[Stream Secure] Notification permission not granted");
      return;
    }
    const payment = calculateNextPayment(sub);
    await chrome.notifications.create(`p01-payment-${sub.id}`, {
      type: "basic",
      iconUrl: chrome.runtime.getURL("icons/icon-128.png"),
      title: "Stream Secure Payment Due",
      message: `${sub.name}: ${payment.amount.toFixed(4)} ${sub.tokenSymbol} is ready to send`,
      priority: 2,
      requireInteraction: true
    });
  } catch (error) {
    console.error("[Stream Secure] Failed to create notification:", error);
  }
}
(_b = (_a = chrome.notifications) == null ? void 0 : _a.onButtonClicked) == null ? void 0 : _b.addListener((notificationId, buttonIndex) => {
  var _a2, _b2;
  if (!notificationId.startsWith("p01-payment-")) return;
  const subscriptionId = notificationId.replace("p01-payment-", "");
  if (buttonIndex === 0) {
    (_b2 = (_a2 = chrome.action).openPopup) == null ? void 0 : _b2.call(_a2).catch(() => {
      chrome.windows.create({
        url: chrome.runtime.getURL("popup.html#/subscriptions/" + subscriptionId),
        type: "popup",
        width: 375,
        height: 640,
        focused: true
      });
    });
  } else if (buttonIndex === 1) {
    pendingPayments = pendingPayments.filter((p) => p.subscriptionId !== subscriptionId);
    chrome.notifications.clear(notificationId);
  }
});
(_d = (_c = chrome.notifications) == null ? void 0 : _c.onClicked) == null ? void 0 : _d.addListener((notificationId) => {
  if (!notificationId.startsWith("p01-payment-")) return;
  const subscriptionId = notificationId.replace("p01-payment-", "");
  chrome.windows.create({
    url: chrome.runtime.getURL("popup.html#/subscriptions/" + subscriptionId),
    type: "popup",
    width: 375,
    height: 640,
    focused: true
  });
  chrome.notifications.clear(notificationId);
});
async function getWalletPublicKey() {
  var _a2;
  try {
    const result = await chrome.storage.local.get("p01-wallet");
    const walletState = result["p01-wallet"] ? JSON.parse(result["p01-wallet"]) : null;
    return ((_a2 = walletState == null ? void 0 : walletState.state) == null ? void 0 : _a2.publicKey) || null;
  } catch (error) {
    console.error("[Background] Failed to get wallet public key:", error);
    return null;
  }
}
function generateApprovalId() {
  return crypto.randomUUID();
}
async function createApprovalRequest(request, tabId) {
  const id = generateApprovalId();
  const approval = {
    ...request,
    id,
    createdAt: Date.now(),
    tabId
  };
  state.pendingApprovals.set(id, approval);
  await chrome.storage.session.set({ currentApproval: approval });
  console.log("[Background] Created approval request:", id, approval);
  return id;
}
async function openApprovalPopup(type) {
  var _a2;
  let path = "/connect";
  if (type === "transaction" || type === "signMessage") {
    path = "/approve";
  } else if (type === "subscription") {
    path = "/approve-subscription";
  }
  await chrome.storage.session.set({
    pendingApprovalPath: path,
    pendingApprovalTimestamp: Date.now()
  });
  try {
    if ((_a2 = chrome.action) == null ? void 0 : _a2.openPopup) {
      await chrome.action.openPopup();
      console.log("[Background] Opened native extension popup");
    }
  } catch (e) {
    console.log("[Background] Could not auto-open popup, user should click extension icon:", e);
    try {
      await chrome.action.setBadgeText({ text: "!" });
      await chrome.action.setBadgeBackgroundColor({ color: "#39c5bb" });
    } catch {
    }
  }
}
async function resolveApproval(requestId, approved, data, reason) {
  const approval = state.pendingApprovals.get(requestId);
  if (!approval) {
    console.warn("[Background] Approval not found:", requestId);
    return;
  }
  state.pendingApprovals.delete(requestId);
  await chrome.storage.session.remove("currentApproval");
  if (approval.tabId) {
    try {
      await chrome.tabs.sendMessage(approval.tabId, {
        type: "APPROVAL_RESULT",
        requestId,
        approved,
        data,
        reason
      });
    } catch (error) {
      console.error("[Background] Failed to send approval result to tab:", error);
    }
  }
  if (approval.origin) {
    await notifyOrigin(approval.origin, {
      type: "APPROVAL_RESULT",
      requestId,
      approved,
      data,
      reason
    });
  }
}
async function notifyOrigin(origin, message) {
  try {
    const tabs = await chrome.tabs.query({ url: `${origin}/*` });
    for (const tab of tabs) {
      if (tab.id) {
        try {
          await chrome.tabs.sendMessage(tab.id, message);
        } catch {
        }
      }
    }
  } catch (error) {
    console.error("[Background] Failed to notify origin:", error);
  }
}
async function handleConnect(payload, sender) {
  var _a2, _b2;
  const origin = payload._origin || payload.origin || "";
  const tabId = (_a2 = sender.tab) == null ? void 0 : _a2.id;
  const walletResult = await chrome.storage.local.get("p01-wallet");
  const walletState = walletResult["p01-wallet"] ? JSON.parse(walletResult["p01-wallet"]) : null;
  const isWalletInitialized = !!((_b2 = walletState == null ? void 0 : walletState.state) == null ? void 0 : _b2.encryptedSeedPhrase);
  if (!isWalletInitialized) {
    await openApprovalPopup("connect");
    return { error: "Wallet not initialized. Please create a wallet first." };
  }
  const isConnected = await isSiteConnected(origin);
  if (isConnected) {
    const publicKey = await getWalletPublicKey();
    if (publicKey) {
      return { publicKey };
    }
  }
  const requestId = await createApprovalRequest(
    {
      type: "connect",
      origin,
      originName: payload.title || (origin ? new URL(origin).hostname : "Unknown"),
      originIcon: payload.icon,
      payload: {}
    },
    tabId
  );
  await openApprovalPopup("connect");
  return new Promise((resolve) => {
    const approval = state.pendingApprovals.get(requestId);
    if (approval) {
      approval.resolve = async () => {
        const publicKey = await getWalletPublicKey();
        if (publicKey) {
          resolve({ publicKey });
        } else {
          resolve({ error: "Wallet not initialized" });
        }
      };
      approval.reject = (error) => {
        resolve({ error: error.message });
      };
    }
    setTimeout(() => {
      if (state.pendingApprovals.has(requestId)) {
        state.pendingApprovals.delete(requestId);
        resolve({ error: "Connection request timeout" });
      }
    }, 3e5);
  });
}
async function handleConnectSilent(payload, _sender) {
  const origin = payload._origin || "";
  const isConnected = await isSiteConnected(origin);
  if (!isConnected) {
    return { connected: false };
  }
  const publicKey = await getWalletPublicKey();
  if (!publicKey) {
    return { connected: false };
  }
  return { connected: true, publicKey };
}
async function handleDisconnect(payload, _sender) {
  const origin = payload._origin || payload.origin || "";
  await removeConnectedSite(origin);
  await notifyOrigin(origin, { type: "DISCONNECT_NOTIFICATION" });
  return { success: true };
}
async function handleIsConnected(payload, _sender) {
  const origin = payload._origin || "";
  const isConnected = await isSiteConnected(origin);
  return { connected: isConnected };
}
async function handleGetAccounts(payload, _sender) {
  const origin = payload._origin || "";
  const isConnected = await isSiteConnected(origin);
  if (!isConnected) {
    return { accounts: [] };
  }
  const publicKey = await getWalletPublicKey();
  if (!publicKey) {
    return { accounts: [] };
  }
  return { accounts: [publicKey] };
}
async function handleSignMessage(payload, sender) {
  var _a2;
  const origin = payload._origin || "";
  const tabId = (_a2 = sender.tab) == null ? void 0 : _a2.id;
  const hasPermission = await siteHasPermission(origin, "requestTransaction");
  if (!hasPermission) {
    return { error: "Not connected or missing permission" };
  }
  const requestId = await createApprovalRequest(
    {
      type: "signMessage",
      origin,
      payload: {
        message: payload.message,
        displayText: payload.displayText
      }
    },
    tabId
  );
  await openApprovalPopup("signMessage");
  return { pending: true, requestId };
}
async function handleSignTransaction(payload, sender) {
  var _a2;
  const origin = payload._origin || "";
  const tabId = (_a2 = sender.tab) == null ? void 0 : _a2.id;
  const hasPermission = await siteHasPermission(origin, "requestTransaction");
  if (!hasPermission) {
    return { error: "Not connected or missing permission" };
  }
  const requestId = await createApprovalRequest(
    {
      type: "transaction",
      origin,
      payload: {
        transaction: payload.transaction,
        isPrivate: payload.isPrivate
      }
    },
    tabId
  );
  await openApprovalPopup("transaction");
  return { pending: true, requestId };
}
async function handleSignAllTransactions(payload, sender) {
  var _a2;
  const origin = payload._origin || "";
  const tabId = (_a2 = sender.tab) == null ? void 0 : _a2.id;
  const hasPermission = await siteHasPermission(origin, "requestTransaction");
  if (!hasPermission) {
    return { error: "Not connected or missing permission" };
  }
  const requestId = await createApprovalRequest(
    {
      type: "transaction",
      origin,
      payload: {
        transactions: payload.transactions,
        isMultiple: true
      }
    },
    tabId
  );
  await openApprovalPopup("transaction");
  return { pending: true, requestId };
}
async function handleSignAndSendTransaction(payload, sender) {
  var _a2;
  const origin = payload._origin || "";
  const tabId = (_a2 = sender.tab) == null ? void 0 : _a2.id;
  const hasPermission = await siteHasPermission(origin, "requestTransaction");
  if (!hasPermission) {
    return { error: "Not connected or missing permission" };
  }
  const requestId = await createApprovalRequest(
    {
      type: "transaction",
      origin,
      payload: {
        transaction: payload.transaction,
        options: payload.options,
        sendAfterSign: true
      }
    },
    tabId
  );
  await openApprovalPopup("transaction");
  return { pending: true, requestId };
}
async function handleCreateSubscription(payload, sender) {
  var _a2;
  const origin = payload._origin || "";
  const tabId = (_a2 = sender.tab) == null ? void 0 : _a2.id;
  const hasPermission = await siteHasPermission(origin, "requestSubscription");
  if (!hasPermission) {
    return { error: "Not connected or missing subscription permission" };
  }
  const requestId = await createApprovalRequest(
    {
      type: "subscription",
      origin,
      payload: {
        recipient: payload.recipient,
        merchantName: payload.merchantName,
        merchantLogo: payload.merchantLogo,
        tokenMint: payload.tokenMint,
        amountPerPeriod: payload.amountPerPeriod,
        periodSeconds: payload.periodSeconds,
        maxPeriods: payload.maxPeriods,
        description: payload.description,
        amountNoise: payload.amountNoise,
        timingNoise: payload.timingNoise,
        useStealthAddress: payload.useStealthAddress
      }
    },
    tabId
  );
  await openApprovalPopup("subscription");
  return { pending: true, requestId };
}
async function handleGetSubscriptions(_payload, _sender) {
  var _a2;
  const result = await chrome.storage.local.get("p01-subscriptions");
  const storedData = result["p01-subscriptions"] ? JSON.parse(result["p01-subscriptions"]) : null;
  return { subscriptions: ((_a2 = storedData == null ? void 0 : storedData.state) == null ? void 0 : _a2.subscriptions) || [] };
}
async function handleCancelSubscription(payload, _sender) {
  var _a2;
  const subscriptionId = payload.subscriptionId;
  if (!subscriptionId) {
    return { error: "Subscription ID required" };
  }
  const result = await chrome.storage.local.get("p01-subscriptions");
  const storedData = result["p01-subscriptions"] ? JSON.parse(result["p01-subscriptions"]) : null;
  if ((_a2 = storedData == null ? void 0 : storedData.state) == null ? void 0 : _a2.subscriptions) {
    storedData.state.subscriptions = storedData.state.subscriptions.map(
      (sub) => sub.id === subscriptionId ? { ...sub, isActive: false } : sub
    );
    await chrome.storage.local.set({
      "p01-subscriptions": JSON.stringify(storedData)
    });
  }
  return { success: true };
}
async function handleApproveRequest(payload) {
  const { requestId, data, permissions } = payload;
  if (!requestId) {
    return { error: "Request ID required" };
  }
  const approval = state.pendingApprovals.get(requestId);
  if (!approval) {
    return { error: "Request not found" };
  }
  if (approval.type === "connect") {
    const dapp = {
      origin: approval.origin,
      name: approval.originName || (approval.origin ? new URL(approval.origin).hostname : "Unknown"),
      icon: approval.originIcon,
      connectedAt: Date.now(),
      permissions: permissions || ["viewBalance", "requestTransaction", "requestSubscription"]
    };
    await addConnectedSite(dapp);
  }
  if (approval.resolve) {
    approval.resolve(data);
  }
  await resolveApproval(requestId, true, data);
  return { success: true };
}
async function handleRejectRequest(payload) {
  const { requestId, reason } = payload;
  if (!requestId) {
    return { error: "Request ID required" };
  }
  const approval = state.pendingApprovals.get(requestId);
  if (!approval) {
    return { error: "Request not found" };
  }
  if (approval.reject) {
    approval.reject(new Error(reason || "User rejected"));
  }
  await resolveApproval(requestId, false, void 0, reason || "User rejected");
  return { success: true };
}
async function handleGetConnectedDapps() {
  const sites = await getConnectedSites();
  return Object.values(sites);
}
async function handleDisconnectDapp(payload) {
  const origin = payload.origin || "";
  await removeConnectedSite(origin);
  await notifyOrigin(origin, { type: "DISCONNECT_NOTIFICATION" });
  return { success: true };
}
async function handleGetApprovalQueue() {
  return Array.from(state.pendingApprovals.values()).map(
    ({ resolve, reject, tabId, ...rest }) => rest
  );
}
registerHandler("GET_ACCOUNTS", async () => {
  const publicKey = await getWalletPublicKey();
  if (!publicKey) {
    throw new Error("Wallet not initialized");
  }
  return { publicKey };
});
registerHandler("GET_APPROVAL_QUEUE", async () => {
  return await handleGetApprovalQueue();
});
registerHandler("APPROVE_REQUEST", async (payload) => {
  return await handleApproveRequest(payload);
});
registerHandler("REJECT_REQUEST", async (payload) => {
  return await handleRejectRequest(payload);
});
registerHandler("GET_CONNECTED_DAPPS", async () => {
  return await handleGetConnectedDapps();
});
registerHandler("DISCONNECT_DAPP", async (payload) => {
  return await handleDisconnectDapp(payload);
});
registerHandler("GET_SUBSCRIPTIONS", async () => {
  var _a2;
  const result = await chrome.storage.local.get("p01-subscriptions");
  const storedData = result["p01-subscriptions"] ? JSON.parse(result["p01-subscriptions"]) : null;
  return ((_a2 = storedData == null ? void 0 : storedData.state) == null ? void 0 : _a2.subscriptions) || [];
});
registerHandler("GET_PENDING_PAYMENTS", async () => {
  return pendingPayments;
});
registerHandler("PAYMENT_PROCESSED", async (payload) => {
  var _a2;
  const { subscriptionId, success, payment } = payload;
  pendingPayments = pendingPayments.filter((p) => p.subscriptionId !== subscriptionId);
  (_a2 = chrome.notifications) == null ? void 0 : _a2.clear(`p01-payment-${subscriptionId}`);
  if (success && payment) {
    try {
      await chrome.notifications.create(`p01-payment-success-${subscriptionId}`, {
        type: "basic",
        iconUrl: chrome.runtime.getURL("icons/icon-128.png"),
        title: "Payment Sent",
        message: `Successfully paid ${payment.amount.toFixed(4)} SOL`,
        priority: 1
      });
      setTimeout(() => {
        var _a3;
        (_a3 = chrome.notifications) == null ? void 0 : _a3.clear(`p01-payment-success-${subscriptionId}`);
      }, 5e3);
    } catch {
    }
  }
  return { success: true };
});
registerHandler("WALLET_UNLOCKED", async () => {
  await chrome.storage.session.set({ "p01-wallet-unlocked": true });
  await processPendingPayments();
  return { success: true };
});
registerHandler("WALLET_LOCKED", async () => {
  await chrome.storage.session.remove("p01-wallet-unlocked");
  return { success: true };
});
registerHandler("CONNECT", async (payload, sender) => {
  return await handleConnect(payload, sender);
});
registerHandler("CONNECT_SILENT", async (payload, sender) => {
  return await handleConnectSilent(payload);
});
registerHandler("DISCONNECT", async (payload, sender) => {
  return await handleDisconnect(payload);
});
registerHandler("IS_CONNECTED", async (payload, sender) => {
  return await handleIsConnected(payload);
});
registerHandler("SIGN_MESSAGE", async (payload, sender) => {
  return await handleSignMessage(payload, sender);
});
registerHandler("SIGN_TRANSACTION", async (payload, sender) => {
  return await handleSignTransaction(payload, sender);
});
registerHandler("SIGN_ALL_TRANSACTIONS", async (payload, sender) => {
  return await handleSignAllTransactions(payload, sender);
});
registerHandler("SIGN_AND_SEND_TRANSACTION", async (payload, sender) => {
  return await handleSignAndSendTransaction(payload, sender);
});
registerHandler("CREATE_SUBSCRIPTION", async (payload, sender) => {
  return await handleCreateSubscription(payload, sender);
});
registerHandler("CANCEL_SUBSCRIPTION", async (payload, sender) => {
  return await handleCancelSubscription(payload);
});
chrome.runtime.onMessage.addListener(
  (message, sender, sendResponse) => {
    if (message.source === "p01-content") {
      const handler = async () => {
        try {
          const payload = message.payload || {};
          switch (message.type) {
            case "CONNECT":
              return await handleConnect(payload, sender);
            case "CONNECT_SILENT":
              return await handleConnectSilent(payload, sender);
            case "DISCONNECT":
              return await handleDisconnect(payload, sender);
            case "IS_CONNECTED":
              return await handleIsConnected(payload, sender);
            case "GET_ACCOUNTS":
              return await handleGetAccounts(payload, sender);
            case "SIGN_MESSAGE":
              return await handleSignMessage(payload, sender);
            case "SIGN_TRANSACTION":
              return await handleSignTransaction(payload, sender);
            case "SIGN_ALL_TRANSACTIONS":
              return await handleSignAllTransactions(payload, sender);
            case "SIGN_AND_SEND_TRANSACTION":
              return await handleSignAndSendTransaction(payload, sender);
            case "CREATE_SUBSCRIPTION":
              return await handleCreateSubscription(payload, sender);
            case "GET_SUBSCRIPTIONS":
              return await handleGetSubscriptions(payload, sender);
            case "CANCEL_SUBSCRIPTION":
              return await handleCancelSubscription(payload, sender);
            default:
              return { error: `Unknown message type: ${message.type}` };
          }
        } catch (error) {
          console.error(`[Background] Error handling ${message.type}:`, error);
          return { error: error instanceof Error ? error.message : "Unknown error" };
        }
      };
      handler().then(sendResponse);
      return true;
    }
    return false;
  }
);
chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === "keepAlive") {
    console.log("[Protocol 01] Keepalive ping");
  } else if (alarm.name === SUBSCRIPTION_CHECK_ALARM) {
    checkDueSubscriptions();
  }
});
initBackgroundMessageListener();
initSubscriptionScheduler();
console.log("[Protocol 01] Background service worker initialized");
console.log("[Stream Secure] Privacy-enhanced recurring payments enabled");
chrome.alarms.create("keepAlive", { periodInMinutes: 1 });
setTimeout(() => {
  checkDueSubscriptions();
}, 5e3);
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === "install") {
    console.log("[Protocol 01] Extension installed");
  } else if (details.reason === "update") {
    console.log(
      "[Protocol 01] Extension updated to version",
      chrome.runtime.getManifest().version
    );
  }
});
