(function(){const CHANNEL = "p01-extension-channel";
const ALLOWED_MESSAGE_TYPES = [
  // Connection
  "CONNECT",
  "CONNECT_SILENT",
  "DISCONNECT",
  "IS_CONNECTED",
  // Accounts
  "GET_ACCOUNTS",
  // Signing
  "SIGN_MESSAGE",
  "SIGN_TRANSACTION",
  "SIGN_ALL_TRANSACTIONS",
  "SIGN_AND_SEND_TRANSACTION",
  // Protocol 01 specific
  "SEND_PRIVATE",
  "GENERATE_STEALTH_ADDRESS",
  "CREATE_SUBSCRIPTION",
  "GET_SUBSCRIPTIONS",
  "CANCEL_SUBSCRIPTION"
];
function injectProviderScript() {
  try {
    const script = document.createElement("script");
    script.src = chrome.runtime.getURL("inject.js");
    script.id = "protocol01-provider";
    const container = document.head || document.documentElement;
    container.insertBefore(script, container.firstChild);
    script.onload = () => {
      script.remove();
    };
    script.onerror = (error) => {
      console.error("[Protocol 01] Failed to inject provider script:", error);
      script.remove();
    };
  } catch (error) {
    console.error("[Protocol 01] Error injecting provider script:", error);
  }
}
function sendToInject(type, payload, requestId, error) {
  window.postMessage(
    {
      source: "p01-content",
      channel: CHANNEL,
      type,
      payload,
      requestId,
      error
    },
    "*"
  );
}
async function forwardToBackground(type, payload, requestId, origin) {
  var _a;
  try {
    const enrichedPayload = {
      ...typeof payload === "object" && payload !== null ? payload : {},
      _origin: origin,
      _tabId: void 0
      // Will be filled by background
    };
    const response = await chrome.runtime.sendMessage({
      source: "p01-content",
      type,
      payload: enrichedPayload,
      requestId
    });
    if (response == null ? void 0 : response.error) {
      sendToInject(type + "_RESPONSE", null, requestId, response.error);
    } else if ((response == null ? void 0 : response.data) !== void 0) {
      if ((_a = response.data) == null ? void 0 : _a.error) {
        sendToInject(type + "_RESPONSE", null, requestId, response.data.error);
      } else {
        sendToInject(type + "_RESPONSE", response.data, requestId);
      }
    } else {
      sendToInject(type + "_RESPONSE", response, requestId);
    }
  } catch (error) {
    const errorMessage = error instanceof Error ? error.message : "Unknown error";
    console.error(`[Protocol 01] Error forwarding ${type}:`, errorMessage);
    sendToInject(type + "_RESPONSE", null, requestId, errorMessage);
  }
}
function handleInjectMessage(event) {
  if (event.source !== window) return;
  const data = event.data;
  if (!data || typeof data !== "object") return;
  if (data.source !== "p01-inject") return;
  if (data.channel !== CHANNEL) return;
  const { type, payload, requestId } = data;
  if (!type || typeof type !== "string") return;
  if (!requestId || typeof requestId !== "string") return;
  if (!ALLOWED_MESSAGE_TYPES.includes(type)) {
    console.warn(`[Protocol 01] Unknown message type: ${type}`);
    sendToInject(type + "_RESPONSE", null, requestId, `Unknown message type: ${type}`);
    return;
  }
  const origin = window.location.origin;
  forwardToBackground(type, payload, requestId, origin);
}
function handleBackgroundMessage(message, _sender, sendResponse) {
  if (message.type === "APPROVAL_RESULT") {
    window.postMessage(
      {
        source: "p01-content",
        channel: CHANNEL,
        type: "APPROVAL_RESULT",
        payload: {
          approved: message.approved,
          reason: message.reason,
          data: message.data
        },
        requestId: message.requestId
      },
      "*"
    );
    sendResponse({ received: true });
    return false;
  }
  if (message.type === "ACCOUNT_CHANGED") {
    window.postMessage(
      {
        source: "p01-content",
        channel: CHANNEL,
        type: "ACCOUNT_CHANGED",
        payload: message.payload,
        requestId: ""
      },
      "*"
    );
    sendResponse({ received: true });
    return false;
  }
  if (message.type === "DISCONNECT_NOTIFICATION") {
    window.postMessage(
      {
        source: "p01-content",
        channel: CHANNEL,
        type: "DISCONNECT",
        payload: {},
        requestId: ""
      },
      "*"
    );
    sendResponse({ received: true });
    return false;
  }
  sendResponse({ received: true });
  return false;
}
function initialize() {
  const protocol = window.location.protocol;
  if (protocol !== "https:" && protocol !== "http:") {
    return;
  }
  injectProviderScript();
  window.addEventListener("message", handleInjectMessage);
  chrome.runtime.onMessage.addListener(handleBackgroundMessage);
  window.postMessage(
    {
      source: "p01-content",
      channel: CHANNEL,
      type: "CONTENT_SCRIPT_READY",
      payload: { version: "0.1.0" },
      requestId: ""
    },
    "*"
  );
  console.log("[Protocol 01] Content script initialized");
}
initialize();
})()
